﻿Option Strict On
Option Explicit On

Imports System.Media
Imports System.IO
Imports System.Drawing.Drawing2D

Public Class Form1

#Region "Fields"
    ' --- Game parameters ---
    Private hitRange As Integer = 10
    Private Const treasureRadius As Integer = 10
    Private treasureCount As Integer = 5
    Private rand As New Random()

    ' --- Game state ---
    Private foundTreasures As New List(Of Point)()
    Private treasures As New List(Of Point)()
    Private playerClickPoints As New List(Of Point)()

    ' --- UI & assets ---
    Private mapWindow As MapForm
    Private digSound As SoundPlayer
    Private foundSound As SoundPlayer
    Private shovelIcon As Image
    Private treasureImages As New List(Of Image)()
    Private treasureImageAssignments As New Dictionary(Of Point, Image)()

    ' --- Timers / animation ---
    Private gameTimer As New Timer()
    Private bonusFlashTimer As New Timer()
    Private bonusPulseTimer As New Timer()
    Private digAnimationTimer As New Timer()
    Private digAnimations As New List(Of (Position As Point, Radius As Integer, StepCount As Integer))()

    ' --- Gameplay metrics ---
    Private elapsedSeconds As Integer = 0
    Private score As Integer = 0
    Private basePoints As Integer = 1000
    Private currentDifficulty As String = "Normal"

    ' --- Leveling & clicks ---
    Private currentLevel As Integer = 1
    Private maxLevels As Integer = 5
    Private levelInProgress As Boolean = False
    Private maxClicks As Integer = 30
    Private remainingClicks As Integer = 30
    Private levelDifficultyFactor As Double = 1.0
    Private lastLevelPerformance As String = String.Empty

    ' --- UI controls (created at runtime) ---
    Private WithEvents gameArea As Panel
    Private lblTime As Label
    Private lblScore As Label
    Private lblLeft As Label
    Private lblClicks As Label
    Private lblBonus As Label

    Private uiFont As New Font("Segoe UI", 11, FontStyle.Regular)
    Private uiLabelFont As New Font("Segoe UI", 12, FontStyle.Bold)

    ' --- Persistence ---
    Private highScoreFile As String = "highscores.txt"

    ' --- Bonus / pulse animation state ---
    Private bonusFlashVisible As Boolean = True
    Private bonusPulseStep As Integer = 150
    Private pulseIncreasing As Boolean = True
    Private criticalBonusThreshold As Integer = 1000

#End Region

#Region "Form lifecycle"
    Private Sub Form1_Load(sender As Object, e As EventArgs) Handles MyBase.Load
        InitializeForm()
        CreateUI()
        WireUpTimers()
        LoadAssets()
        ApplySavedSettings()

        UpdateAllDisplays()
    End Sub

    Private Sub InitializeForm()
        Me.Text = "Treasure Hunt!"
        ''Me.StartPosition = FormStartPosition.CenterScreen
        ''Me.CenterToScreen()
        Me.FormBorderStyle = FormBorderStyle.FixedSingle
        Me.MaximizeBox = False
        Me.BackColor = Color.FromArgb(248, 236, 207)
        Me.Font = uiFont
        Me.Size = New Size(960, 640)
        Me.DoubleBuffered = True
    End Sub
#End Region

#Region "UI Construction"
    Private Sub CreateUI()
        ' TOP PANEL (controls)
        Dim topPanel As New Panel With {
            .Dock = DockStyle.Top,
            .Height = 62,
            .Padding = New Padding(10),
            .BackColor = Color.FromArgb(225, 204, 169)
        }
        Controls.Add(topPanel)

        ' LEFT: Buttons & difficulty
        Dim leftFlow As New FlowLayoutPanel With {
            .Dock = DockStyle.Left,
            .Size = New Drawing.Size(550, 40),
            .FlowDirection = FlowDirection.LeftToRight
        }
        topPanel.Controls.Add(leftFlow)

        Dim btnBury As New Button With {.Text = "Bury", .Width = 90, .AutoSize = True}
        Dim btnReset As New Button With {.Text = "Reset", .Width = 90, .AutoSize = True}
        Dim btnMap As New Button With {.Text = "Map", .Width = 90, .AutoSize = True}
        Dim btnScores As New Button With {.Text = "Scores", .Width = 110, .AutoSize = True}

        AddHandler btnBury.Click, AddressOf BuryTreasure
        AddHandler btnReset.Click, AddressOf ResetGame
        AddHandler btnMap.Click, AddressOf ShowMap
        AddHandler btnScores.Click, AddressOf ShowHighScores

        leftFlow.Controls.AddRange(New Control() {btnBury, btnReset, btnMap, btnScores})

        Dim cmbDifficulty As New ComboBox With {
            .DropDownStyle = ComboBoxStyle.DropDownList,
            .Width = 120
        }
        cmbDifficulty.Items.AddRange(New String() {"Easy", "Normal", "Hard", "Expert"})
        cmbDifficulty.SelectedItem = "Normal"
        AddHandler cmbDifficulty.SelectedIndexChanged, AddressOf DifficultyChanged
        leftFlow.Controls.Add(cmbDifficulty)

        ' RIGHT: status labels
        Dim rightFlow As New FlowLayoutPanel With {
            .Dock = DockStyle.Right,
            .Size = New Drawing.Size(350, 40),
            .FlowDirection = FlowDirection.LeftToRight
        }
        topPanel.Controls.Add(rightFlow)

        lblTime = New Label With {.AutoSize = True, .Font = uiFont}
        lblScore = New Label With {.AutoSize = True, .Font = uiFont}
        lblLeft = New Label With {.AutoSize = True, .Font = uiFont}
        lblClicks = New Label With {.AutoSize = True, .Font = uiFont}
        lblBonus = New Label With {.AutoSize = True, .Font = uiLabelFont}
        lblBonus.ForeColor = Color.DarkGreen

        rightFlow.Controls.AddRange(New Control() {lblTime, lblScore, lblLeft, lblClicks, lblBonus})

        ' GAME AREA (center)
        gameArea = New Panel With {
            .Dock = DockStyle.Fill,
            .BackColor = Color.FromArgb(244, 222, 179)
        }
        Controls.Add(gameArea)

        ' Enable double buffering on the panel (non-public property)
        Dim doubleBufferProperty = gameArea.GetType().GetProperty("DoubleBuffered", Reflection.BindingFlags.Instance Or Reflection.BindingFlags.NonPublic)
        If doubleBufferProperty IsNot Nothing Then
            doubleBufferProperty.SetValue(gameArea, True, Nothing)
        End If

        ' Assign paint and mouse handlers
        AddHandler gameArea.Paint, AddressOf GameArea_Paint
        AddHandler gameArea.MouseClick, AddressOf GameArea_MouseClick
    End Sub
#End Region

#Region "Assets & Timers"
    Private Sub LoadAssets()
        ' Sounds
        Try
            If File.Exists("dig.wav") Then
                digSound = New SoundPlayer("dig.wav")
                digSound.LoadAsync()
            End If
            If File.Exists("found.wav") Then
                foundSound = New SoundPlayer("found.wav")
                foundSound.LoadAsync()
            End If
        Catch
            ' Ignore failures — fallback to system sounds
        End Try

        ' Shovel icon
        If File.Exists("shovel.png") Then
            Try
                shovelIcon = Image.FromFile("shovel.png")
            Catch
            End Try
        End If

        ' Treasure images (optional)
        Dim imageFiles As String() = {"coin.png", "gem.png", "chest.png"}
        For Each imgPath In imageFiles
            If File.Exists(imgPath) Then
                Try
                    treasureImages.Add(Image.FromFile(imgPath))
                Catch
                End Try
            End If
        Next
    End Sub

    Private Sub WireUpTimers()
        gameTimer.Interval = 1000
        AddHandler gameTimer.Tick, AddressOf GameTimer_Tick

        bonusFlashTimer.Interval = 300
        AddHandler bonusFlashTimer.Tick, AddressOf BonusFlash_Tick

        bonusPulseTimer.Interval = 80
        AddHandler bonusPulseTimer.Tick, AddressOf BonusPulse_Tick

        digAnimationTimer.Interval = 40
        AddHandler digAnimationTimer.Tick, AddressOf DigAnimation_Tick
        digAnimationTimer.Start()
    End Sub
#End Region

#Region "Gameplay: Start / Reset / Map"
    Private Sub BuryTreasure(sender As Object, e As EventArgs)
        If levelInProgress Then
            MessageBox.Show("You’re already playing! Find all treasures to advance.")
            Return
        End If

        If currentLevel = 1 Then
            ' fresh session
            score = 0
            elapsedSeconds = 0
        End If

        treasures.Clear()
        foundTreasures.Clear()
        levelInProgress = True
        playerClickPoints.Clear()

        bonusFlashTimer.Stop()
        bonusPulseTimer.Stop()
        lblBonus.ForeColor = Color.DarkGreen
        lblBonus.Visible = True

        ' Difficulty-adjusted parameters
        Dim areaWidth As Integer = Math.Max(200, gameArea.ClientSize.Width - 20)
        Dim areaHeight As Integer = Math.Max(120, gameArea.ClientSize.Height - 20)

        Dim levelFactor As Double = levelDifficultyFactor
        Dim adjustedTreasureCount As Integer = Math.Max(3, CInt(treasureCount * levelFactor))
        Dim adjustedMaxClicks As Integer = Math.Max(10, CInt(maxClicks / (levelFactor * 0.9)))
        remainingClicks = adjustedMaxClicks

        For i As Integer = 1 To adjustedTreasureCount
            treasures.Add(New Point(rand.Next(50, areaWidth - 50), rand.Next(20, areaHeight - 20)))
        Next

        ' Randomize treasure images
        treasureImageAssignments.Clear()
        If treasureImages.Count > 0 Then
            For Each t In treasures
                treasureImageAssignments(t) = treasureImages(rand.Next(treasureImages.Count))
            Next
        End If

        UpdateAllDisplays()

        MessageBox.Show($"Level {currentLevel} started!{vbCrLf}Find {adjustedTreasureCount} treasures.", "Next Level")

        PlayFoundSound()
        gameTimer.Start()
        gameArea.Invalidate()
    End Sub

    Private Sub ResetGame(sender As Object, e As EventArgs)
        levelDifficultyFactor = 1.0
        lastLevelPerformance = String.Empty
        currentLevel = 1
        levelInProgress = False
        treasures.Clear()
        foundTreasures.Clear()
        playerClickPoints.Clear()
        score = 0
        elapsedSeconds = 0

        bonusFlashTimer.Stop()
        bonusPulseTimer.Stop()
        lblBonus.ForeColor = Color.DarkGreen
        lblBonus.Visible = True

        gameTimer.Stop()
        remainingClicks = maxClicks

        UpdateAllDisplays()
        gameArea.Invalidate()
    End Sub

    Private Sub ShowMap(sender As Object, e As EventArgs)
        If mapWindow Is Nothing OrElse mapWindow.IsDisposed Then
            mapWindow = New MapForm()
        End If
        UpdateMap()
        mapWindow.Show()
        mapWindow.BringToFront()
    End Sub

    Private Sub UpdateMap()
        If mapWindow IsNot Nothing AndAlso Not mapWindow.IsDisposed Then
            Dim sourceRect As New Rectangle(0, 0, gameArea.ClientSize.Width, gameArea.ClientSize.Height)
            mapWindow.UpdateMap(treasures.Except(foundTreasures).ToList(), foundTreasures.ToList(), playerClickPoints.ToList(), sourceRect)
        End If
    End Sub
#End Region

#Region "Timers & Bonus UI"
    Private Sub GameTimer_Tick(sender As Object, e As EventArgs)
        elapsedSeconds += 1
        UpdateAllDisplays()
    End Sub

    Private Sub BonusFlash_Tick(sender As Object, e As EventArgs)
        bonusFlashVisible = Not bonusFlashVisible
        lblBonus.Visible = bonusFlashVisible
    End Sub

    Private Sub BonusPulse_Tick(sender As Object, e As EventArgs)
        If pulseIncreasing Then
            bonusPulseStep += 15
            If bonusPulseStep >= 255 Then
                bonusPulseStep = 255
                pulseIncreasing = False
            End If
        Else
            bonusPulseStep -= 15
            If bonusPulseStep <= 100 Then
                bonusPulseStep = 100
                pulseIncreasing = True
            End If
        End If
        lblBonus.ForeColor = Color.FromArgb(bonusPulseStep, 255, 60, 60)
    End Sub
#End Region

#Region "Input: Mouse Clicks & Digging"
    Private Sub GameArea_MouseClick(sender As Object, e As MouseEventArgs)
        ' Record click for map
        playerClickPoints.Add(e.Location)
        UpdateMap()

        If treasures.Count = 0 Then
            PlayDigSound()
            Return
        End If

        ' Add dig animation
        digAnimations.Add((e.Location, 5, 0))

        ' Deduct click
        remainingClicks -= 1
        If remainingClicks < 0 Then remainingClicks = 0
        UpdateClicksDisplay()

        If remainingClicks <= 0 Then
            MessageBox.Show("No more digs left! Round over.", "Out of Clicks")
            EndGame()
            Return
        End If

        PlayDigSound()

        For Each t In treasures
            Dim dx As Integer = e.X - t.X
            Dim dy As Integer = e.Y - t.Y
            Dim distance As Double = Math.Sqrt(dx * dx + dy * dy)
            If distance < hitRange AndAlso Not foundTreasures.Contains(t) Then
                foundTreasures.Add(t)

                Dim decay As Integer = 10
                Dim pts As Integer = basePoints - (elapsedSeconds * decay)
                If pts < 50 Then pts = 50
                score += pts

                ' Reward extra digs
                remainingClicks += 3
                If remainingClicks > maxClicks Then remainingClicks = maxClicks
                UpdateClicksDisplay()

                PlayFoundSound()
                MessageBox.Show($"You found a treasure! +{pts} points", "Success")

                gameArea.Invalidate()
                UpdateMap()
                UpdateAllDisplays()

                If foundTreasures.Count >= treasures.Count Then
                    EndGame()
                End If
                Exit For
            End If
        Next
    End Sub
#End Region

#Region "Animations"
    Private Sub DigAnimation_Tick(sender As Object, e As EventArgs)
        If digAnimations.Count > 0 Then
            For i As Integer = 0 To digAnimations.Count - 1
                Dim anim = digAnimations(i)
                anim.Radius += 4
                anim.StepCount += 1
                digAnimations(i) = anim
            Next

            digAnimations.RemoveAll(Function(a) a.StepCount > 15)
            gameArea.Invalidate(New Rectangle(0, 0, gameArea.ClientSize.Width, gameArea.ClientSize.Height))
        End If
    End Sub
#End Region

#Region "Drawing"
    Private Sub GameArea_Paint(sender As Object, e As PaintEventArgs)
        Dim g As Graphics = e.Graphics
        g.SmoothingMode = SmoothingMode.AntiAlias

        ' Background grid / border
        Using borderPen As New Pen(Color.Brown, 2)
            g.DrawRectangle(borderPen, 0, 0, gameArea.ClientSize.Width - 1, gameArea.ClientSize.Height - 1)
        End Using

        ' Draw found treasures
        For Each t In foundTreasures
            If treasureImageAssignments.ContainsKey(t) Then
                Dim img As Image = treasureImageAssignments(t)
                Dim imgSize As Integer = 28
                g.DrawImage(img, t.X - imgSize \ 2, t.Y - imgSize \ 2, imgSize, imgSize)
            Else
                Using b As New SolidBrush(Color.Gold)
                    g.FillEllipse(b, t.X - treasureRadius, t.Y - treasureRadius, treasureRadius * 2, treasureRadius * 2)
                End Using
                g.DrawEllipse(Pens.Black, t.X - treasureRadius, t.Y - treasureRadius, treasureRadius * 2, treasureRadius * 2)
            End If
        Next

        ' Dig animations
        For Each anim In digAnimations
            Dim alpha As Integer = 255 - (anim.StepCount * 15)
            If alpha < 0 Then alpha = 0
            Using pen As New Pen(Color.FromArgb(alpha, 139, 69, 19), 2)
                g.DrawEllipse(pen, anim.Position.X - anim.Radius, anim.Position.Y - anim.Radius, anim.Radius * 2, anim.Radius * 2)
            End Using

            If shovelIcon IsNot Nothing Then
                Dim iconSize As Integer = 24
                g.DrawImage(shovelIcon, anim.Position.X - iconSize \ 2, anim.Position.Y - iconSize \ 2, iconSize, iconSize)
            End If
        Next

        ' Draw player clicks (small semi-transparent markers)
        For Each p In playerClickPoints
            Using pb As New SolidBrush(Color.FromArgb(110, Color.Black))
                g.FillEllipse(pb, p.X - 3, p.Y - 3, 6, 6)
            End Using
        Next
    End Sub
#End Region

#Region "End of Round / Scoring"
    Private Sub EndGame()
        gameTimer.Stop()
        levelInProgress = False

        ' Performance analysis
        Dim avgTargetTime As Integer = 30 + (currentLevel * 10)
        Dim speedScore As Double = avgTargetTime / Math.Max(1, elapsedSeconds)
        Dim clickEfficiency As Double = remainingClicks / Math.Max(1, maxClicks)
        Dim performanceScore As Double = (speedScore * 0.6) + (clickEfficiency * 0.4)

        If performanceScore > 1.2 Then
            levelDifficultyFactor *= 1.25
            lastLevelPerformance = "Excellent (Speed & Efficiency)"
        ElseIf performanceScore > 0.9 Then
            levelDifficultyFactor *= 1.15
            lastLevelPerformance = "Good (Balanced)"
        ElseIf performanceScore > 0.7 Then
            levelDifficultyFactor *= 1.05
            lastLevelPerformance = "Fair"
        Else
            levelDifficultyFactor *= 0.9
            lastLevelPerformance = "Needs Improvement"
        End If

        levelDifficultyFactor = Math.Max(0.8, Math.Min(2.5, levelDifficultyFactor))
        UpdateDifficultyDisplay()

        If currentLevel < maxLevels Then
            MessageBox.Show($"You cleared Level {currentLevel}!{vbCrLf}Performance: {lastLevelPerformance}{vbCrLf}Next Level Difficulty: {levelDifficultyFactor:F2}×{vbCrLf}Get ready for Level {currentLevel + 1}!", "Level Complete")
            currentLevel += 1
            BuryTreasure(Nothing, Nothing)
            Return
        End If

        Dim playerName As String = InputBox("You found all the treasures! Enter your name:", "Game Over", "Player")
        If String.IsNullOrWhiteSpace(playerName) Then playerName = "Anonymous"

        Dim baseScore As Integer = score

        ' Nonlinear time bonus
        Dim maxBonus As Integer = 4000
        Dim timeConstant As Double = 20.0
        Dim timeBonus As Integer = CInt(maxBonus * Math.Exp(-elapsedSeconds / timeConstant))

        ' Click bonus
        Dim clickBonusPerDig As Integer = If(currentDifficulty = "Easy", 30, If(currentDifficulty = "Normal", 40, If(currentDifficulty = "Hard", 50, 60)))
        Dim clickBonus As Integer = remainingClicks * clickBonusPerDig

        Dim finalScore As Integer = baseScore + timeBonus + clickBonus

        ' Save high score
        Dim recordLine As String = $"{playerName},{finalScore},{elapsedSeconds},{DateTime.Now},{currentDifficulty}"
        Try
            File.AppendAllText(highScoreFile, recordLine & Environment.NewLine)
        Catch ex As Exception
            MessageBox.Show("Could not save high score: " & ex.Message)
        End Try

        ' Show high scores form (if available)
        Try
            Dim hs As New HighScoresForm(highScoreFile, playerName, finalScore)
            hs.ShowDialog()
        Catch
            ' ignore if form not present
        End Try

        Dim msg As String = $"Your Score: {finalScore}{vbCrLf}Base Score: {baseScore}{vbCrLf}Time Bonus: +{timeBonus}{vbCrLf}Click Bonus: +{clickBonus} ({remainingClicks} digs left){vbCrLf}Time: {elapsedSeconds} seconds"
        MessageBox.Show(msg, "Congratulations!", MessageBoxButtons.OK, MessageBoxIcon.Information)

        remainingClicks = maxClicks
        currentLevel = 1
        ResetGame(Nothing, Nothing)
    End Sub
#End Region

#Region "UI Helpers"
    Private Sub UpdateAllDisplays()
        UpdateTimeDisplay()
        UpdateScoreDisplay()
        UpdateLeftDisplay()
        UpdateClicksDisplay()
        UpdateBonusDisplay()
    End Sub

    Private Sub UpdateTimeDisplay()
        lblTime.Text = $"Time: {elapsedSeconds}s"
    End Sub

    Private Sub UpdateScoreDisplay()
        lblScore.Text = $"Score: {score}"
    End Sub

    Private Sub UpdateLeftDisplay()
        lblLeft.Text = $"Treasures: {Math.Max(0, treasures.Count - foundTreasures.Count)}"
    End Sub

    Private Sub UpdateClicksDisplay()
        lblClicks.Text = $"Clicks Left: {remainingClicks}"
    End Sub

    Private Sub UpdateBonusDisplay()
        Dim maxBonus As Integer = 4000
        Dim timeConstant As Double = 20.0
        Dim currentBonus As Integer = CInt(maxBonus * Math.Exp(-elapsedSeconds / timeConstant))

        lblBonus.Text = $"Bonus: {currentBonus}"

        If currentBonus < criticalBonusThreshold Then
            If Not bonusPulseTimer.Enabled Then
                pulseIncreasing = True
                bonusPulseStep = 150
                bonusPulseTimer.Start()
            End If
        Else
            lblBonus.ForeColor = Color.DarkGreen
            If bonusPulseTimer.Enabled Then bonusPulseTimer.Stop()
        End If
    End Sub

    Private Sub UpdateDifficultyDisplay()
        ' Update a label showing difficulty factor in the top panel (if present)
        For Each c As Control In Controls
            If TypeOf c Is Panel Then
                For Each child As Control In c.Controls
                    For Each lbl In child.Controls.OfType(Of Label)()
                        If lbl.Text.StartsWith("Difficulty") Then
                            lbl.Text = $"Difficulty x{levelDifficultyFactor:F2}"
                            Return
                        End If
                    Next
                Next
            End If
        Next
    End Sub
#End Region

#Region "Sounds"
    Private Sub PlayDigSound()
        Try
            If digSound IsNot Nothing Then digSound.Play() Else System.Media.SystemSounds.Beep.Play()
        Catch
        End Try
    End Sub

    Private Sub PlayFoundSound()
        Try
            If foundSound IsNot Nothing Then foundSound.Play() Else System.Media.SystemSounds.Asterisk.Play()
        Catch
        End Try
    End Sub
#End Region

#Region "Difficulty & Settings"
    Private Sub DifficultyChanged(sender As Object, e As EventArgs)
        Dim cmb As ComboBox = DirectCast(sender, ComboBox)
        currentDifficulty = cmb.SelectedItem.ToString()
        SetDifficulty(currentDifficulty)
        Try
            My.Settings.DifficultyLevel = currentDifficulty
            My.Settings.Save()
        Catch
        End Try
    End Sub

    Private Sub SetDifficulty(level As String)
        Select Case level
            Case "Easy"
                treasureCount = 3
                hitRange = 30
                basePoints = 800
                maxClicks = 60
            Case "Normal"
                treasureCount = 5
                hitRange = 20
                basePoints = 1000
                maxClicks = 50
            Case "Hard"
                treasureCount = 7
                hitRange = 15
                basePoints = 1200
                maxClicks = 40
            Case "Expert"
                treasureCount = 10
                hitRange = 10
                basePoints = 1500
                maxClicks = 30
        End Select
        hitRange = hitRange * 2
        remainingClicks = maxClicks
        UpdateClicksDisplay()
    End Sub

    Private Sub ApplySavedSettings()
        Try
            Dim savedDifficulty As String = My.Settings.DifficultyLevel
            If Not String.IsNullOrEmpty(savedDifficulty) Then
                currentDifficulty = savedDifficulty
                SetDifficulty(currentDifficulty)
            End If
        Catch
        End Try
    End Sub
#End Region

#Region "High Scores"
    Private Sub ShowHighScores(sender As Object, e As EventArgs)
        Try
            Dim hs As New HighScoresForm(highScoreFile)
            hs.ShowDialog()
        Catch ex As Exception
            MessageBox.Show("High scores unavailable: " & ex.Message)
        End Try
    End Sub
#End Region

End Class

